/*************************************************************************
**************************************************************************
Fonction de calcul des OR par classes et de leur intervalle de confiance après une commande mfp, fp ou fracpoly


dernière mise à jour 28 novembre 2024

**************************************************************************
**************************************************************************/



/**************************************************************************
 Ce programme est appelé par la commande ORcl_pf après un modèle de régression logistique où une variable X est remplacée par un polynôme fractionnaire. Ce modèle doit être créé par une des commandes mfp, fp ou fracpoly.

La syntaxe est ORcl_pf,ref(#) cl(numlist) 

- ne pas oublier la virgule après ORcl_pf

- ref(#) est le centre de la classe de référence et # doit être remplacé par une seule valeur

- cl est la liste des centres de classes, séparés par un blanc ou une virgule, pour lesquelles on veut avoir l'OR et son IC (il doit y avoir au moins une classe).

Le programme est limité aux cas où le polynôme fractionnaire de X est de degré ≤ 2 (une ou deux puissances)
***********************************************************************/

/****************************************************
*****************************************************
Exemple
mfp : logistic acc age
ORcl_pf, ref(27) cl(17 22 32 37 42)

Résultat
Modélisation : mfp : logistic acc age
variables dans le modèle final : fracpoly: logistic acc age 3 3
OR et IC pour la variable age

ref = 2.7   classe = 17   0R = 0.573    95% CI = [0.380 - 0.865]
ref = 2.7   classe = 22   0R = 0.801    95% CI = [0.657 - 0.976]
ref = 2.7   classe = 32   0R = 0.964    95% CI = [0.846 - 1.098]
ref = 2.7   classe = 37   0R = 0.606    95% CI = [0.509 - 0.721]
ref = 2.7   classe = 42   0R = 0.206    95% CI = [0.145 - 0.291]
*****************************************************
****************************************************/


capt program drop ORcl_pf
program define ORcl_pf


syntax [varlist], ref(real) cl(numlist min=1)


if "`e(fp_cmd)'"=="fracpoly" {
	tokenize `e(fp_k1)'
	local p1 `1'
	local p2 `2'

	local varx1 `e(fp_x1)'  // x contient le nom de la variable x1
	tempvar x
	qui gen `x'=`varx1'  // pour ne pas changer la variable x1 (varx1) du fichier en cas de scaling

	** nom des variables dans le modèle final mfp
	local fpx1="I`varx1'__1"
	local fpx2="I`varx1'__2"
	local dim : word count `e(fp_k1)' // dimension du PF
}

else if substr("`e(fp_cmd)'",1,2)=="fp"{
	local p1 = e(fp_fp)[1,1]
	local p2 = e(fp_fp)[1,2]
	local varx1 `e(fp_variable)'
	tempvar x
	qui gen `x'= `e(fp_variable)'
	** nom des variables dans le modèle final fp
	local fpx1="`e(fp_variable)'_1"
	local fpx2="`e(fp_variable)'_2"
	local dim=`e(fp_dimension)'  // dimension du PF
}
else {
	di as err "La commande précédente doit être mfp, fp ou fracpoly"
	exit 198
}

if `dim'>2  {
	di as err "le nombre de puissance pour la variable " as res "`varx1'" as err " doit être ≤ 2"
	exit 198
}

****** Dénomination par cli des classes successives de x1 dont on veut les OR 

local ncl : word count `cl'

tokenize `cl'
foreach i of numlist 1/`ncl' {
	local cl`i' ``i''
}

****** Changement d'échelle pour X (scaling)
** nb : l'option center de fp ou de mfp n'est pas prise en compte car elle ne change que la constante et donc pas l'OR

if "`e(fp_cmd)'"=="fracpoly" {  // si la commande utilisée est mfp ou fracpoly
	/*
	règle dans mfp (trouvée dans frac_xo.ado et adaptée). Cette règle figure aussi dans le help détaillé de fp, écrite de façon différente mais équivalente
	il est possible que l'adaptation ait des lacunes pour certaines puissances des FP (en particulier pour modèle linéaire)
	l'option noscaling n'existe pas dans mfp et il y a une option scale dans fp
	*/
	
	qui sum `x' 
	tempname s
	scalar `s'=r(max)-r(min)
	local scale = 10^(sign(log10(`s'))*int(abs(log10(`s'))))
	qui replace `x'=`x'/`scale'
	local ref=`ref'/`scale'
	}
else if substr("`e(fp_cmd)'",1,2)=="fp" {  // si la commande utilisée est fp
	qui replace `x'= (`x'-`e(fp_scale_a)')/`e(fp_scale_b)'
}
******

*** Annonce des résultats
 
if "`e(fp_cmd2)'"=="mfp"{
	di as text _n(1) "Modélisation : " "`e(cmdline)'"
	di as text "variables dans le modèle final : " "`e(fracpoly)'"
}
else if substr("`e(fp_cmd)'",1,2)=="fp" {
	di as text _n(1) "Modélisation : " "`e(fp_cmdline)'"
}
else if "`e(fp_cmd)'"=="fracpoly" {
	di as text _n(1) "Modélisation : " "`e(cmdline)'" ", puissance(s) de la variable `varx1' : `e(fp_pwrs)'"
}


di as text "OR et IC pour la variable " "`varx1'" _n(1)



****** Boucle sur les valeurs de cl

local ref_or=`ref' // utile pour l'affichage des résultats

foreach i of numlist 1/`ncl' {
	local cl_or=`cl`i''
	if  "`e(fp_cmd)'"=="fracpoly" {
		local cl=`cl`i''/`scale'
	}
	else if substr("`e(fp_cmd)'",1,2)=="fp" {  // si la commande utilisée est fp
	local cl= (`cl`i''-`e(fp_scale_a)')/`e(fp_scale_b)'
	}
	
	****** Calcul des coefficients qui doivent figurer dans lincom
	** nb : la 2ème puissance (`2') est toujours inférieure ou égale à la 1ère (`1'). D'où les if et else successifs.
	
	if `p1'!=0 {
		local d1=(`cl')^`p1'-(`ref')^`p1'
	}
	else {
		local d1=ln(`cl')-ln(`ref')
	}
	
	if "`p2'"=="" {
		local d2=0
	}
	else if `p2'!=0 & `p2'!=`p1' {
		local d2=(`cl')^`p2'-(`ref')^`p2'
	}
	else if `p2'!=0 & `p2'==`p1' {
		local d2=(`cl')^`p1'*ln(`cl')-(`ref')^`p1'*ln(`ref')
	}
	else if `p2'==0 & `p1'==0 {
		local d2=ln(`cl')*ln(`cl')-ln(`ref')*ln(`ref')
	}
	
	***********

	
	****** Ecriture du résultat (avec 3 décimales pour les OR et IC)
	
	 if "`p2'"!="" {
	 *qui lincom `d1'*I`varx1'__1+`d2'*I`varx1'__2
	 qui lincom `d1'*`fpx1'+`d2'*`fpx2'
	 }
	 else {
	 qui lincom `d1'*`fpx1'
	 }

	
		disp as res "ref = " `ref_or' _s(3) "classe = " `cl_or'  ///
		_s(3) "0R = "  %-6.3f `r(estimate)' _s(3)  "`r(level)'% CI = ["  %-6.3f `r(lb)' "-"  %6.3f `r(ub)' "]"
		
	******
	
} // fin de la boucle sur les valeurs de cl


end	
	


